package com.iptv.iptvapp

import android.app.Application
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.viewModelScope
import com.iptv.iptvapp.data.Channel
import com.iptv.iptvapp.data.IptvApiService
import com.iptv.iptvapp.data.SerialRepository
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.launch

class ChannelViewModel(application: Application) : AndroidViewModel(application) {
    private val apiService = IptvApiService.create()
    private val serialRepository = SerialRepository(application)

    private val _channels = MutableStateFlow<List<Channel>>(emptyList())
    private val _filteredChannels = MutableStateFlow<List<Channel>>(emptyList())
    val channels: StateFlow<List<Channel>> = _filteredChannels.asStateFlow()

    private val _searchQuery = MutableStateFlow("")
    val searchQuery: StateFlow<String> = _searchQuery.asStateFlow()

    private val _selectedChannel = MutableStateFlow<Channel?>(null)
    val selectedChannel: StateFlow<Channel?> = _selectedChannel.asStateFlow()

    private val _isLoading = MutableStateFlow(false)
    val isLoading: StateFlow<Boolean> = _isLoading.asStateFlow()

    private val _error = MutableStateFlow<String?>(null)
    val error: StateFlow<String?> = _error.asStateFlow()

    private val _serialNumber = MutableStateFlow<String?>(null)
    val serialNumber: StateFlow<String?> = _serialNumber.asStateFlow()

    private val _expiryMessage = MutableStateFlow<String?>(null)
    val expiryMessage: StateFlow<String?> = _expiryMessage.asStateFlow()

    private val _currentPackId = MutableStateFlow<Int?>(null)
    val currentPackId: StateFlow<Int?> = _currentPackId.asStateFlow()

    val packageIds = listOf(1, 3, 4, 5, 7, 8, 9, 10, 12, 13, 14, 15, 16, 17, 18, 22)

    init {
        viewModelScope.launch {
            val savedSerial = serialRepository.serialNumber.first()
            if (!savedSerial.isNullOrBlank()) {
                _serialNumber.value = savedSerial
                checkActivation(savedSerial)
            } else {
                _serialNumber.value = null
            }
        }
    }

    fun updateSerial(newSerial: String) {
        viewModelScope.launch {
            serialRepository.saveSerial(newSerial)
            _serialNumber.value = if (newSerial.isBlank()) null else newSerial
            if (newSerial.isNotBlank()) {
                checkActivation(newSerial)
            } else {
                _expiryMessage.value = null
            }
        }
    }

    private fun checkActivation(serial: String) {
        viewModelScope.launch {
            _isLoading.value = true
            try {
                val response = apiService.checkActivation(
                    login = "79461301202032",
                    uid = "00:00:f5:10:84:87",
                    serial = serial
                )
                _expiryMessage.value = response.auth?.firstOrNull()?.msg
            } catch (e: Exception) {
                _error.value = "Activation check failed: ${e.message}"
            } finally {
                _isLoading.value = false
            }
        }
    }

    fun fetchChannels(packId: Int) {
        val serial = _serialNumber.value ?: return
        viewModelScope.launch {
            _isLoading.value = true
            _error.value = null
            _currentPackId.value = packId
            try {
                val response = apiService.getChannels(
                    login = "79461301202032",
                    uid = "00:00:f5:10:84:87",
                    serial = serial,
                    packId = packId
                )
                val newChannels = response.channels ?: emptyList()
                _channels.value = newChannels
                filterChannels(_searchQuery.value)
            } catch (e: Exception) {
                _error.value = "Failed to fetch channels: ${e.message}"
            } finally {
                _isLoading.value = false
            }
        }
    }

    fun onSearchQueryChanged(query: String) {
        _searchQuery.value = query
        if (query.length >= 3) {
            // If currentPackId is null (we are in package list), search across all packages
            if (_currentPackId.value == null) {
                searchAllPackages(query)
            } else {
                filterChannels(query)
            }
        } else {
            if (_currentPackId.value == null) {
                // Clear search results if query too short in global mode
                // But do NOT clear channels list if we are inside a package
                 _filteredChannels.value = emptyList()
            } else {
                // If inside a package, show all channels for that package
                _filteredChannels.value = _channels.value
            }
        }
    }

    private fun searchAllPackages(query: String) {
        val serial = _serialNumber.value ?: return
        viewModelScope.launch {
            _isLoading.value = true
            _error.value = null
            
            // Temporary list to collect results
            val allFoundChannels = mutableListOf<Channel>()
            
            try {
                // Use a coroutine scope to launch parallel requests for faster search
                // Note: Be mindful of API rate limits if any
                // Since packageIds is small, we can probably do them in parallel chunks or all at once
                
                // For simplicity and safety, let's just do sequential for now to avoid complexity,
                // but optimize the loop
                
                 packageIds.forEach { packId ->
                     try {
                        val response = apiService.getChannels(
                            login = "79461301202032",
                            uid = "00:00:f5:10:84:87",
                            serial = serial,
                            packId = packId
                        )
                        val packChannels = response.channels ?: emptyList()
                        // Filter locally
                        val matching = packChannels.filter { 
                            it.name.contains(query, ignoreCase = true) 
                        }
                        allFoundChannels.addAll(matching)
                     } catch (e: Exception) {
                         // Continue searching other packages even if one fails
                     }
                }
                
                // Update the UI with found channels
                _filteredChannels.value = allFoundChannels
                
            } catch (e: Exception) {
                _error.value = "Global search error: ${e.message}"
            } finally {
                _isLoading.value = false
            }
        }
    }

    private fun filterChannels(query: String) {
        if (query.length >= 3) {
            _filteredChannels.value = _channels.value.filter {
                it.name.contains(query, ignoreCase = true)
            }
        } else {
            _filteredChannels.value = _channels.value
        }
    }

    fun selectChannel(channel: Channel?) {
        _selectedChannel.value = channel
    }

    fun goBackToPackages() {
        _currentPackId.value = null
        _channels.value = emptyList()
        _filteredChannels.value = emptyList() // Also clear filtered channels
        _selectedChannel.value = null
    }
}
